import sys, os
from gi.repository import GLib
from dbus.mainloop.glib import DBusGMainLoop

sys.path.insert(0, '/data/velib_python')
from vedbus import VeDbusService

BUS = 'com.victronenergy.batterycells'
PIPE_PATH = '/tmp/canvoltages'

def main():
    if not os.path.exists(PIPE_PATH):
        open(PIPE_PATH, 'w').close()

    DBusGMainLoop(set_as_default=True)
    service = VeDbusService(BUS, register=False)

    # Basic info paths
    service.add_path('/Management/ProcessName', os.path.basename(__file__))
    service.add_path('/Management/DeviceInstance', 0)
    service.add_path('/Management/Connection', 'Battery cell daemon')
    service.add_path('/ProductId', 123)
    service.add_path('/ProductName', 'CellMonitor')
    service.add_path('/Connected', 1)

    registered_modules = set()
    service.register()

    # helper checks (use float compare with small epsilon)
    def is_voltage_placeholder_val(v):
        try:
            return abs(float(v) - 65.535) < 1e-6
        except Exception:
            return False

    def is_temp_placeholder_val(v):
        try:
            return abs(float(v) - (-1.0)) < 1e-6
        except Exception:
            return False

    # factories to avoid late-binding problems
    def make_voltage_gettext(path):
        def gettext(p, v):
            try:
                if v is None:
                    return "Not connected"
                if is_voltage_placeholder_val(v):
                    return "Not connected"
                return f"{float(v):.3f} V"
            except Exception:
                return "Not connected"
        return gettext

    def make_temp_gettext(path):
        def gettext(p, v):
            try:
                if v is None:
                    return "Not connected"
                if is_temp_placeholder_val(v):
                    return "Not connected"
                return f"{float(v):.1f} °C"
            except Exception:
                return "Not connected"
        return gettext

    def update():
        try:
            with open(PIPE_PATH, 'r') as f:
                lines = f.readlines()
        except Exception as e:
            print("❌ Could not read pipe:", e)
            return True

        if len(lines) < 1:
            return True

        try:
            num_modules = int(lines[0].strip())
            print(f"🆔 Active modules reported: {num_modules}")
        except Exception as e:
            print("⚠️ Invalid module count:", e)
            return True

        for i in range(1, num_modules + 1):
            mod_name = f'Module{i}'

            if mod_name not in registered_modules:
                print(f"🧩 Registering {mod_name}")

                service.add_path(f'/{mod_name}/CanStatus', num_modules)

                for j in range(16):
                    cell_voltage_path = f'/{mod_name}/Cell{str(j).zfill(2)}/Voltage'
                    service.add_path(cell_voltage_path, 0.0,
                                     gettextcallback=make_voltage_gettext(cell_voltage_path))

                for j in range(8):
                    cell_temp_path = f'/{mod_name}/Cell{str(j).zfill(2)}/Temperature'
                    service.add_path(cell_temp_path, 0.0,
                                     gettextcallback=make_temp_gettext(cell_temp_path))

                bms_current_path = f'/{mod_name}/Bms/Current'
                bms_temp_path = f'/{mod_name}/Bms/Temperature'
                service.add_path(bms_current_path, 0.0,
                                 gettextcallback=lambda p, v, cp=bms_current_path: f"{v:.1f} A" if v is not None else "Not connected")
                service.add_path(bms_temp_path, 0.0,
                                 gettextcallback=lambda p, v, tp=bms_temp_path: f"{v:.1f} °C" if v is not None else "Not connected")

                registered_modules.add(mod_name)

        # Update module data
        for i in range(1, num_modules + 1):
            mod_name = f'Module{i}'
            try:
                line = lines[i].strip()
            except IndexError:
                print(f"⚠️ No data line for {mod_name}")
                continue

            values = line.split()
            if len(values) < 18:
                print(f"⚠️ {mod_name}: Only {len(values)} values (need ≥18)")
                continue

            service[f'/{mod_name}/CanStatus'] = num_modules

            for j in range(16):
                try:
                    raw = values[j]
                except IndexError:
                    continue
                try:
                    v = float(raw)
                except Exception:
                    print(f"⚠️ {mod_name}: Cannot parse voltage at index {j} ('{raw}')")
                    continue
                service[f'/{mod_name}/Cell{str(j).zfill(2)}/Voltage'] = v
                if is_voltage_placeholder_val(raw):
                    print(f"🔋 {mod_name} Cell{j} Voltage: Not connected (placeholder {raw})")
                else:
                    print(f"🔋 {mod_name} Cell{j} Voltage: {v:.3f} V")

            temp_values = values[16:-2]
            for j, rawt in enumerate(temp_values):
                try:
                    t = float(rawt)
                except Exception:
                    print(f"⚠️ {mod_name}: Cannot parse temp at index {j} ('{rawt}')")
                    continue
                service[f'/{mod_name}/Cell{str(j).zfill(2)}/Temperature'] = t
                if is_temp_placeholder_val(rawt):
                    print(f"🌡️ {mod_name} Cell{j} Temp: Not connected (placeholder {rawt})")
                else:
                    print(f"🌡️ {mod_name} Cell{j} Temp: {t:.1f} °C")

            try:
                current_str = values[-2]
                current = float(current_str)
                service[f'/{mod_name}/Bms/Current'] = current
                print(f"🔄 {mod_name} BMS Current: {current:.1f} A")
            except (IndexError, ValueError):
                bad = values[-2] if len(values) >= 2 else 'Not connected'
                print(f"⚠️ {mod_name}: Invalid BMS current value: {bad}")

            try:
                bms_temp_str = values[-1]
                bms_temp = float(bms_temp_str)
                service[f'/{mod_name}/Bms/Temperature'] = bms_temp
                print(f"🌡️ {mod_name} BMS Temp: {bms_temp:.1f} °C")
            except (IndexError, ValueError):
                bad = values[-1] if values else 'Not connected'
                print(f"⚠️ {mod_name}: Invalid BMS temp value: {bad}")

        return True

    GLib.timeout_add_seconds(5, update)
    print(f"✅ Monitoring {PIPE_PATH} ...")
    GLib.MainLoop().run()

if __name__ == '__main__':
    main()