#!/bin/sh
FILE="/tmp/canvoltages"
PACKS=15

to_voltage() {
    val=$(( (0x$1 << 8) + 0x$2 ))
    awk "BEGIN {printf \"%.3f\", $val / 1000}"
}

set_var() { eval "$1=\"\$2\""; }
get_var() { eval "printf '%s' \"\${$1}\""; }

init_pack() {
    p=$1
    i=0
    while [ $i -le 15 ]; do
        set_var "v_p${p}_c${i}" "0.000"
        i=$((i+1))
    done
    i=0
    while [ $i -le 7 ]; do
        set_var "t_p${p}_${i}" "0.0"
        i=$((i+1))
    done
    set_var "cur_p${p}" "0.0"
    set_var "bms_p${p}" "25.0"
    set_var "mask_p${p}" 0
}

write_file() {
    {
        # First line: id000_val
        printf "%s\n" "$id000_val"

        # Write only as many modules as id000_val indicates
        p=1
        while [ $p -le "$id000_val" ]; do
            line=""
            i=0
            while [ $i -le 15 ]; do
                line="$line$(get_var v_p${p}_c$i) "
                i=$((i+1))
            done
            i=0
            while [ $i -le 7 ]; do
                line="$line$(get_var t_p${p}_$i) "
                i=$((i+1))
            done
            line="$line$(get_var cur_p${p}) $(get_var bms_p${p})"
            printf "%s\n" "$line"
            p=$((p+1))
        done
    } > "$FILE"
}


set_mask_bit() {
    p=$1
    low=$2
    bit=$((1 << (low - 1)))
    cur=$(get_var mask_p${p})
    cur=$((cur | bit))
    set_var "mask_p${p}" "$cur"
}

pack_complete() {
    p=$1
    cur=$(get_var mask_p${p})
    [ "$cur" -eq 63 ]
}

reset_mask() {
    p=$1
    set_var "mask_p${p}" 0
}

# ---------- init ----------
id000_val=0
p=1
while [ $p -le $PACKS ]; do
    init_pack "$p"
    p=$((p+1))
done

write_file
echo "🟢 Listening for CAN data on vecan1..."

while true; do
    candump vecan1 | while read -r iface id len b1 b2 b3 b4 b5 b6 b7 b8; do
        [ -z "$id" ] && continue

        id_clean=$(printf "%s" "$id" | tr -d '[:space:]' | tr '[:lower:]' '[:upper:]')
        case "$id_clean" in
            *[!0123456789ABCDEF]*) continue ;;
        esac

        id_dec=$((16#$id_clean))

        if [ "$id_dec" -eq 0 ]; then
            id000_val=$((0x${b1:-0}))
            write_file
            continue
        fi

        low=$(( id_dec & 0xF ))
        high=$(( (id_dec >> 4) & 0xFF ))
        pack=$((high + 1))

        [ $pack -lt 1 ] || [ $pack -gt $PACKS ] && continue
        [ $low -lt 1 ] || [ $low -gt 6 ] && continue

        case $low in
            1|2|3|4)
                gi=$((low - 1))
                base=$((gi * 4))
                c0=$(to_voltage "${b2:-00}" "${b1:-00}")
                c1=$(to_voltage "${b4:-00}" "${b3:-00}")
                c2=$(to_voltage "${b6:-00}" "${b5:-00}")
                c3=$(to_voltage "${b8:-00}" "${b7:-00}")
                set_var "v_p${pack}_c$((base+0))" "$c0"
                set_var "v_p${pack}_c$((base+1))" "$c1"
                set_var "v_p${pack}_c$((base+2))" "$c2"
                set_var "v_p${pack}_c$((base+3))" "$c3"
                set_mask_bit "$pack" "$low"
                ;;
            5)
                i=0
                while [ $i -le 7 ]; do
                    hex=$(eval "printf '%s' \"\${b$((i+1)):-}\"")
                    dec=0
                    [ -n "$hex" ] && dec=$((0x$hex))
                    [ $dec -ge 128 ] && dec=$((dec - 256))
                    set_var "t_p${pack}_$i" "$(printf '%.1f' "$dec")"
                    i=$((i+1))
                done
                write_file
                set_mask_bit "$pack" 5
                ;;
            6)
                # raw 16-bit current (high byte b2, low byte b1)
                raw_current=$(( (0x${b2:-00} << 8) + 0x${b1:-00} ))
                # interpret as signed 16-bit (two's complement)
                if [ "$raw_current" -ge 32768 ]; then
                    raw_current=$((raw_current - 65536))
                fi
                cur=$(awk "BEGIN {printf \"%.1f\", $raw_current / 10}")
                b3d=$((0x${b3:-00}))
                [ $b3d -ge 128 ] && b3d=$((b3d - 256))
                set_var "cur_p${pack}" "$cur"
                set_var "bms_p${pack}" "$(printf '%.1f' "$b3d")"
                set_mask_bit "$pack" 6
                ;;
        esac

        if [ $low -le 4 ] && pack_complete "$pack"; then
            write_file
            reset_mask "$pack"
        fi

    done
    echo "⚠️ candump exited, restarting in 1 second..."
    sleep 1
done